<?PHP
#
#   FILE:  Plugin.php
#
#   Part of the Metavus digital collections platform
#   Copyright 2016-2020 Edward Almasy and Internet Scout Research Group
#   http://metavus.net
#
# @scout:phpstan

namespace Metavus;

use Exception;
use ScoutLib\StdLib;

/**
 * This class extends the base Plugin class with CWIS-specific functionality.
 */
abstract class Plugin extends \ScoutLib\Plugin
{
    # ---- PUBLIC INTERFACE --------------------------------------------------


    # ---- PROTECTED INTERFACE -----------------------------------------------

    /**
     * Load fields into metadata schema from XML file.  The XML file is
     * assumed to be in install/MetadataSchema--SCHEMANAME.xml under the
     * plugin's directory.
     * @param mixed $Schema Schema or ID of schema to load fields into.
     * @return string|null Error message or NULL if load succeeded.
     * @throws Exception If no XML file found.
     */
    protected function addMetadataFieldsFromXml($Schema)
    {
        # load schema
        if (!($Schema instanceof MetadataSchema)) {
            $Schema = new MetadataSchema($Schema);
        }

        # assemble XML file name
        $PossibleSuffixes = [
            StdLib::singularize($Schema->resourceName()),
            StdLib::pluralize($Schema->resourceName())
        ];
        foreach ($PossibleSuffixes as $Suffix) {
            $FileName = "plugins/".static::getBaseName()
                    ."/install/MetadataSchema--"
                    .str_replace(" ", "", $Suffix).".xml";
            if (is_file($FileName)) {
                $XmlFile = $FileName;
                break;
            }
        }
        if (!isset($XmlFile)) {
            throw new Exception("No XML file found to load metadata fields from"
                    ." for ".$Schema->name()." schema.");
        }

        # load fields from file
        $Result = $Schema->addFieldsFromXmlFile($XmlFile);

        # if load failed
        if ($Result === false) {
            # return error message(s) to caller
            return "Error loading User metadata fields from XML: ".implode(
                " ",
                $Schema->errorMessages("AddFieldsFromXmlFile")
            );
        }

        # report success to caller
        return null;
    }

    /**
     * Delete any metadata fields owned by plugin from specified schema.
     * @param int $SchemaId ID of schema to drop fields from.
     * @return string|null Error message or NULL if drop succeeded.
     */
    protected function deleteMetadataFields(int $SchemaId)
    {
        # load schema
        $Schema = new MetadataSchema($SchemaId);

        # for each field in schema
        foreach ($Schema->getFields() as $FieldId => $Field) {
            # drop field if we own it
            if ($Field->owner() == static::getBaseName()) {
                $Schema->dropField($FieldId);
            }
        }

        # report success to caller
        return null;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------
}
