/**
 * FILE:  CW-Base.js
 *
 * Part of the Metavus digital collections platform
 * CCopyright 2011-2020 Edward Almasy and Internet Scout Research Group
 * http://metavus.net
 *
 * @scout:eslint
 *
 * Support functions for core Metavus.
 */

var cw = {};
/**
 * Extend a subclass to inherit another class' private methods, privileged
 * methods and data members.
 * http://blog.reinpetersen.com/2008/12/work-around-for-common-inheritance.html
 * @param subclass:Object_reference subclass class
 * @param base:Object_reference parent class
 */
cw.extend = function(subclass, base) {
    function Closure(){}
    Closure.prototype = base.prototype;
    subclass.prototype = new Closure();
    subclass.prototype.constructor = subclass;
    subclass.base = base;
};

/**
 * Construct a callback referring the method of a instantiated object.
 * Additional arguments to this function will be used as arguments to the
 * callback. The additional arguments will come before any arguments passed to
 * the callback when executed. For example, if passing the integer "1" as an
 * additional argument, the callback this function returns will pass in the
 * integer "1" as the first argument and then any other arguments passed in at
 * the time it was called.
 * @param object object an instantiated object
 * @param function method a method (function) of the object
 * @param ... additional arguments
 * @return a callback for the method of the object
 */
cw.bindObjectMethodCallback = function(object, method) {
    var parameters = Array.prototype.slice.call(arguments).slice(2);

    return function(){
        return method.apply(
            object,
            parameters.concat(Array.prototype.slice.call(arguments).slice(0)));
    };
};

/**
* Get the base URL for the page.
* @return Returns the base URL for the page.
*/
cw.getBaseUrl = function() {
    var url;

    // if the base URL global is set
    if (typeof CWIS_BASE_URL != "undefined") {
        return CWIS_BASE_URL; // eslint-disable-line no-undef
    }

    // otherwise try to get the base URL from the current URL
    url = window.location.pathname;
    url = url.substring(0, url.lastIndexOf("/")+1);

    return url;
};

/**
* Get the URL for the CWIS router page.
* @return Returns the URL for the CWIS router page.
*/
cw.getRouterUrl = function() {
    return cw.getBaseUrl() + "index.php";
};

/**
* Detect when a user starts tabbing to use keyboard navigation, add a
* focus outline to indicate the focused element.
*/
$(document).on('keydown', function(e) {
    const tabKeyCode = 9;
    if (e.keyCode == tabKeyCode) {
        // add a <style> to the head of the document
        var styleEl = document.createElement('style');
        document.head.appendChild(styleEl);

        // turn on the focus outline in that element
        styleEl.sheet.insertRule(':focus { outline-style: dotted; }');

        // disable keydown handler
        $(document).off('keydown');
    }
});

$(document).ready(function(){
    function adjustElementHeight(el) {
        $(el).height(0);
        $(el).height($(el).prop('scrollHeight'));
    }

    function fixCheckboxSets() {
        $(".mv-checkboxset-item", ".mv-checkboxset").each(function(index, element){
            adjustElementHeight(element);
        });

        $(".mv-checkboxset").each(function(index, element) {
            adjustElementHeight($(element).parent());
        });
    }

    // fix checkbox sets at our initial window size
    fixCheckboxSets();

    // and re-fix them after the window is resized
    $(window).resize(function(){
        fixCheckboxSets();
    });
});
