<?PHP
#
#   FILE:  Home.php
#
#   Part of the Metavus digital collections platform
#   Copyright 2021 Edward Almasy and Internet Scout Research Group
#   http://metavus.net
#
# VALUES PROVIDED to INTERFACE (REQUIRED):
#   $H_FeaturedItems - Items (Record instances) to feature, with item IDs
#       for the index.
#   $H_Collections - Current collections of items (Collection instances),
#       with collection IDs for the index.
# VALUES PROVIDED to INTERFACE (OPTIONAL):
#   $H_NewsItems - News items (Plugin\Blog\Entry instances), with news item
#       IDs for the index.  Only set if news is enabled, Blog plugin is ready,
#       and blog exists with the configured name.
#
# @scout:phpstan

namespace Metavus;

use Metavus\Plugins\Blog\Entry;
use Metavus\Plugins\Blog\EntryFactory;
use ScoutLib\PluginManager;

# ----- LOCAL FUNCTIONS ------------------------------------------------------

/**
 * Load collections for display.
 * @return array Collections (Collection instances), with collection IDs for
 *      the index.
 */
function loadCollections(): array
{
    $User = User::getCurrentUser();
    $CollectionSchemaId = MetadataSchema::getSchemaIdForName("Collections");
    $CFactory = new CollectionFactory();
    $AllCollections = $CFactory->getItems();
    $Collections = [];
    foreach ($AllCollections as $CollectionId => $Collection) {
        if ($Collection->userCanView($User)) {
            $Collections[$CollectionId] = $Collection;
        }
    }
    return $Collections;
}

/**
 * Load items to feature.
 * @return array Items, with item IDs for the index.
 */
function loadFeaturedItems(): array
{
    $SysConfig = SystemConfiguration::getInstance();
    $MaxItems = $SysConfig->getInt("NumResourcesOnHomePage");
    $MaxNumberOfDaysToGoBackForItems = 7300;
    $RFactory = new RecordFactory();
    $FeaturedItems = $RFactory->getRecentlyReleasedRecords(
        ($MaxItems + 1),
        0,
        $MaxNumberOfDaysToGoBackForItems
    );
    return $FeaturedItems;
}

/**
 * Load news items for display.
 * @param string $BlogName Name of blog from which to load news.
 * @return array News items (Entry instances).
 */
function loadNews(string $BlogName): array
{
    $PluginMgr = PluginManager::getInstance();
    $SysConfig = SystemConfiguration::getInstance();

    $BlogPlugin = $PluginMgr->getPlugin("Blog");
    $NewsBlogId = $BlogPlugin->getBlogIdByName($BlogName);

    $MaxNewsItems = $SysConfig->getInt("NumAnnounceOnHomePage");
    $EFactory = new EntryFactory($NewsBlogId);
    $NewsItemIds = $EFactory->getRecordIdsSortedBy(
        "Date of Publication",
        false,
        $MaxNewsItems
    );

    $User = User::getCurrentUser();
    $NewsItems = [];
    foreach ($NewsItemIds as $NewsItemId) {
        $NewsItem = new Entry($NewsItemId);
        if ($NewsItem->userCanView($User)) {
            $NewsItems[$NewsItemId] = $NewsItem;
        }
    }

    return $NewsItems;
}


# ----- MAIN -----------------------------------------------------------------

# configuration settings
$NewsBlogName = "News";

# load news if enabled
$SysConfig = SystemConfiguration::getInstance();
$PluginMgr = PluginManager::getInstance();
if ($SysConfig->getBool("AnnouncementsEnabled")
    && $PluginMgr->pluginReady("Blog")
    && ($PluginMgr->getPlugin("Blog")->getBlogIdByName($NewsBlogName) !== false)) {
    $H_NewsItems = loadNews($NewsBlogName);
}

# load collection info
$H_Collections = loadCollections();

# load featured items
$H_FeaturedItems = loadFeaturedItems();
