<?PHP
#
#   FILE:  OldSystemConfiguration.php
#
#   Part of the Metavus digital collections platform
#   Copyright 2013-2021 Edward Almasy and Internet Scout Research Group
#   http://metavus.net
#
# @scout:phpstan

namespace Metavus;

use ScoutLib\ApplicationFramework;
use ScoutLib\Database;

class OldSystemConfiguration
{
    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
     * Constructs a new SystemConfiguration object.
     */
    public function __construct()
    {
        $this->DB = new Database();
        $this->DB->setValueUpdateParameters("SystemConfiguration");
    }

    /**
     * Gets or sets the top link string system configuration setting.
     * Used by the BrowseResources page to print classification links.
     * @param string $NewValue The new top link string system configuration
     *       setting.  (OPTIONAL)
     * @return string The top link string system configuration setting.
     */
    public function topLinkString(string $NewValue = null)
    {
        return $this->updateValue("TopLinkString", $NewValue);
    }

    /**
     * Gets or sets the top link begin letter system configuration setting.
     * Used by the BrowseResources page to print classification links.
     * @param string $NewValue The new top link begin letter system configuration
     *       setting.  (OPTIONAL)
     * @return string The top link begin letter system configuration setting.
     */
    public function topLinkBeginLetter(string $NewValue = null)
    {
        return $this->updateValue("TopLinkBeginLetter", $NewValue);
    }

    /**
     * Gets or sets the top link end letter system configuration setting.
     * Used by the BrowseResources page to print classification links.
     * @param string $NewValue The new top link end letter system configuration
     *       setting.  (OPTIONAL)
     * @return string The top link end letter system configuration setting.
     */
    public function topLinkEndLetter(string $NewValue = null)
    {
        return $this->updateValue("TopLinkEndLetter", $NewValue);
    }

    /**
     * Gets or sets the admin email system configuration setting.
     * @param string $NewValue The new admin email system configuration
     *       setting.  (OPTIONAL)
     * @return string The admin email system configuration setting.
     */
    public function adminEmail(string $NewValue = null)
    {
        return $this->updateValue("AdminEmail", $NewValue);
    }

    /**
     * Gets or sets the portal name system configuration setting.
     * @param string $NewValue The new portal name system configuration
     *       setting.  (OPTIONAL)
     * @return string The portal name system configuration setting.
     */
    public function portalName(string $NewValue = null)
    {
        return $this->updateValue("PortalName", $NewValue);
    }

    /**
     * Gets or sets the default active UI system configuration setting.
     * @param string $NewValue The new default active UI system configuration
     *       setting.  (OPTIONAL)
     * @return string The default active UI system configuration setting.
     */
    public function defaultActiveUI(string $NewValue = null)
    {
        return $this->updateValue("DefaultActiveUI", $NewValue);
    }

    /**
     * Gets or sets the current theme system configuration setting.
     * @param string $NewValue The new current theme system configuration
     *       setting.  (OPTIONAL)
     * @return string The current theme system configuration setting.
     */
    public function currentTheme(string $NewValue = null)
    {
        return $this->updateValue("CurrentTheme", $NewValue);
    }

    /**
     * Gets or sets the default character set system configuration setting.
     * @param string $NewValue The new default character set system configuration
     *       setting.  (OPTIONAL)
     * @return string The default character set system configuration setting.
     */
    public function defaultCharacterSet(string $NewValue = null)
    {
        return $this->updateValue("DefaultCharacterSet", $NewValue);
    }

    /**
     * Gets or sets the search terms required system configuration setting.
     * Used to determine whether the search logic is AND or OR.
     * @param bool $NewValue The new search terms required system configuration
     *       setting.  (OPTIONAL)
     * @return bool The search terms required system configuration setting.
     */
    public function searchTermsRequired(bool $NewValue = null)
    {
        return (bool)$this->updateValue("SearchTermsRequired", $NewValue);
    }

    /**
     * Gets or sets the show number of resources enabled system configuration
     * setting.  Used on the Home page to display the total number of resources.
     * @param bool $NewValue The new show number of resources enabled system
     *       configuration setting.  (OPTIONAL)
     * @return bool The show number of resources enabled system configuration setting.
     */
    public function showNumResourcesEnabled(bool $NewValue = null)
    {
        return (bool)$this->updateValue("ShowNumResourcesEnabled", $NewValue);
    }

    /**
     * Gets or sets the announcements enabled system configuration setting.
     * @param bool $NewValue The new announcements enabled system configuration
     *       setting.  (OPTIONAL)
     * @return bool The announcements enabled system configuration setting.
     */
    public function announcementsEnabled(bool $NewValue = null)
    {
        return (bool)$this->updateValue("AnnouncementsEnabled", $NewValue);
    }

    /**
     * Gets or sets the comments allow html system configuration setting.
     * @param bool $NewValue The new comments allow html system configuration setting.
     *        (OPTIONAL)
     * #return bool The comments allow html system configuration setting.
     */
    public function commentsAllowHTML(bool $NewValue = null)
    {
        return (bool)$this->updateValue("CommentsAllowHTML", $NewValue);
    }

    /**
     * Gets or sets the allow multiple UIs enabled system configuration setting.
     * Used to determine whether users can use a UI other than the default.
     * @param bool $NewValue The new allow multiple UIs enabled system
     *       configuration setting.  (OPTIONAL)
     * @return bool The allow multiple UIs enabled system configuration setting.
     */
    public function allowMultipleUIsEnabled(bool $NewValue = null)
    {
        return (bool)$this->updateValue("AllowMultipleUIsEnabled", $NewValue);
    }

    /**
     * Gets or sets the resource launches new window enabled system configuration
     * setting. Used by the FullRecorde page to determine whether the URL opens
     * in a new window.
     * @param bool $NewValue The new resource launches new window enabled system
     *       configuration setting.  (OPTIONAL)
     * @return bool The resource launches new window enabled system configuration
     * setting.
     */
    public function resourceLaunchesNewWindowEnabled(bool $NewValue = null)
    {
        return (bool)$this->updateValue("ResourceLaunchesNewWindowEnabled", $NewValue);
    }

    /**
     * Gets or sets the resource ratings enabled system configuration setting.
     * @param bool $NewValue The new resource ratings enabled system configuration
     *       setting.  (OPTIONAL)
     * @return bool The resource ratings enabled system configuration setting.
     */
    public function resourceRatingsEnabled(bool $NewValue = null)
    {
        return (bool)$this->updateValue("ResourceRatingsEnabled", $NewValue);
    }

    /**
     * Gets or sets the incremental keyword search enabled system configuration
     * setting. Used for determining whether searching shows results as you type.
     * @param bool $NewValue The new incremental keyword search enabled system
     *       configuration setting.  (OPTIONAL)
     * @return bool The incremental keyword search enabled system configuration
     *       setting.
     */
    public function incrementalKeywordSearchEnabled(bool $NewValue = null)
    {
        return (bool)$this->updateValue("IncrementalKeywordSearchEnabled", $NewValue);
    }

    /**
     * Gets or sets the search database enabled system configuration setting.
     * @param bool $NewValue The new search database enabled system configuration
     *       setting.  (OPTIONAL)
     * @return bool The search database enabled system configuration setting.
     */
    public function searchDBEnabled(bool $NewValue = null)
    {
        return (bool)$this->updateValue("SearchDBEnabled", $NewValue);
    }

    /**
     * Gets or sets the recommender database enabled system configuration
     * setting.
     * @param bool $NewValue The new recommender database enabled system
     *       configuration setting.  (OPTIONAL)
     * @return bool The recommender database enabled system configuration setting.
     */
    public function recommenderDBEnabled(bool $NewValue = null)
    {
        return (bool)$this->updateValue("RecommenderDBEnabled", $NewValue);
    }

    /**
     * Gets or sets the number of announcements on homepage system configuration
     * setting.
     * @param int $NewValue The new number of announcements on homepage system
     *       configuration setting.  (OPTIONAL)
     * @return int The number of announcements on homepage system configuration
     *       setting.
     */
    public function numAnnounceOnHomePage(int $NewValue = null)
    {
        return (int)$this->updateValue("NumAnnounceOnHomePage", $NewValue);
    }

    /**
     * Gets or sets the number of resources on homepage system configuration
     * setting.
     * @param int $NewValue The new number of resources on homepage system
     *       configuration setting.  (OPTIONAL)
     * @return int The number of resources on homepage system configuration setting.
     */
    public function numResourcesOnHomePage(int $NewValue = null)
    {
        return (int)$this->updateValue("NumResourcesOnHomePage", $NewValue);
    }

    /**
     * Gets or sets the legal notice system configuration setting.
     * @param string $NewValue The new legal notice set system configuration
     *       setting.  (OPTIONAL)
     * @return string The legal notice system configuration setting.
     */
    public function legalNotice(string $NewValue = null)
    {
        return $this->updateValue("LegalNotice", $NewValue);
    }

    /**
     * Gets or sets the default records per page system configuration setting.
     * @param int $NewValue The new default records per page system configuration
     *       setting.  (OPTIONAL)
     * @return int The default records per page system configuration setting.
     */
    public function defaultRecordsPerPage(int $NewValue = null)
    {
        return (int)$this->updateValue("DefaultRecordsPerPage", $NewValue);
    }

    /**
     * Gets or sets the number of resources considered to build search
     * facets.
     * @param int $NewValue Desired setting for the number of
     *        resources to use in computing search facets. (OPTIONAL)
     * @return int The number of resources used in computing search facets.
     */
    public function numResourcesForSearchFacets(int $NewValue = null)
    {
        return (int)$this->updateValue("NumResourcesForSearchFacets", $NewValue);
    }

    /**
     * Gets or sets the number of facets considered excessive, above which we
     * won't display the facets.
     * @param int $NewValue Desired setting for the number of
     *        resources to use in computing search facets. (OPTIONAL)
     * @return int The maximum number of facets per field to display.
     */
    public function maxFacetsPerField(int $NewValue = null)
    {
        return (int)$this->updateValue("MaxFacetsPerField", $NewValue);
    }

    /**
     * Gets or sets the number of classes per browse page system configuration
     * setting.
     * @param int $NewValue The new number of classes per browse page system
     *       configuration setting.  (OPTIONAL)
     * @return int The number of classes per browse page system configuration
     *       setting.
     */
    public function numClassesPerBrowsePage(int $NewValue = null)
    {
        # if new classes per browse page has been specified
        if (($NewValue !== null) &&
            ($NewValue != $this->updateValue("NumClassesPerBrowsePage", null))) {
            # clear all classification links
            $this->topLinkString("");
            $this->DB->query("UPDATE Classifications SET LinkString=''");
        }
        return (int)$this->updateValue("NumClassesPerBrowsePage", $NewValue);
    }

    /**
     * Gets or sets the number of columns per browse page system configuration
     * setting.
     * @param int $NewValue The new number of columns per browse page system
     *       configuration setting.  (OPTIONAL)
     * @return int The number of columns per browse page system configuration
     *       setting.
     */
    public function numColumnsPerBrowsePage(int $NewValue = null)
    {
        return (int)$this->updateValue("NumColumnsPerBrowsePage", $NewValue);
    }

    /**
     * Gets or sets the browsing field ID system configuration setting.
     * Used by the BrowseResources page to determine the default browsing option.
     * @param int $NewValue The new browsing field ID system configuration
     *       setting.  (OPTIONAL)
     * @return int The browsing field ID system configuration setting.
     */
    public function browsingFieldId(int $NewValue = null)
    {
        return (int)$this->updateValue("BrowsingFieldId", $NewValue);
    }

    /**
     * Gets or sets the site keywords system configuration setting.
     * @param string $NewValue The new site keywords system configuration
     *       setting.  (OPTIONAL)
     * @return string The site keywords system configuration setting.
     */
    public function siteKeywords(string $NewValue = null)
    {
        return $this->updateValue("SiteKeywords", $NewValue);
    }

    /**
     * Gets or sets the about text system configuration setting.
     * @param string $NewValue The new about text system configuration
     *       setting.  (OPTIONAL)
     * @return string The about text system configuration setting.
     * @deprecated in CWIS 4.1.2, removed after X-DATE-X
     */
    public function aboutText(string $NewValue = null)
    {
        $GLOBALS["AF"]->logError(
            ApplicationFramework::LOGLVL_WARNING,
            "SystemConfiguration::aboutText() is deprecated, "
            ."about pages should now be handled through the Pages plugin."
        );
        return "";
    }

    /**
     * Gets or sets the contact name system configuration setting.
     * @param string $NewValue The new contact name system configuration
     *       setting.  (OPTIONAL)
     * @return string The contact name system configuration setting.
     */
    public function contactName(string $NewValue = null)
    {
        return $this->updateValue("ContactName", $NewValue);
    }

    /**
     * Gets or sets the contact email system configuration setting.
     * @param string $NewValue The new contact email system configuration
     *       setting.  (OPTIONAL)
     * @return string The contact email system configuration setting.
     */
    public function contactEmail(string $NewValue = null)
    {
        return $this->updateValue("ContactEmail", $NewValue);
    }

    /**
     * Gets or sets the organization system configuration setting.
     * @param string $NewValue The new organization system configuration
     *       setting.  (OPTIONAL)
     * @return string The organization system configuration setting.
     */
    public function organization(string $NewValue = null)
    {
        return $this->updateValue("Organization", $NewValue);
    }

    /**
     * Gets or sets the organization type system configuration setting.
     * @param string $NewValue The new organization type system configuration
     *       setting.  (OPTIONAL)
     * @return string The organization type system configuration setting.
     */
    public function organizationType(string $NewValue = null)
    {
        return $this->updateValue("OrganizationType", $NewValue);
    }

    /**
     * Gets or sets the site type system configuration setting.
     * @param string $NewValue The new site type system configuration
     *       setting.  (OPTIONAL)
     * @return string The site type system configuration setting.
     */
    public function siteType(string $NewValue = null)
    {
        return $this->updateValue("SiteType", $NewValue);
    }

    /**
     * Gets or sets the okay to list site system configuration setting.
     * Used when registering to determine if the site can be included in a public
     * list of CWIS sites.
     * @param bool $NewValue The new okay to list site system configuration
     *       setting.  (OPTIONAL)
     * @return bool The okay to list site system configuration setting.
     */
    public function okayToListSite(bool $NewValue = null)
    {
        return (bool)$this->updateValue("OkayToListSite", $NewValue);
    }

    /**
     * Gets or sets the last registered on system configuration setting.
     * @param string $NewValue The new last registered on system configuration
     *       setting.  (OPTIONAL)
     * @return string The last registered on system configuration setting.
     */
    public function lastRegisteredOn(string $NewValue = null)
    {
        return $this->updateValue("LastRegisteredOn", $NewValue);
    }

    /**
     * Gets or sets the add AW stats script system configuration setting.
     * @param bool $NewValue The new add AW stats script system configuration
     *       setting.  (OPTIONAL)
     * @return bool The add AW stats script system configuration setting.
     */
    public function addAWStatsScript(bool $NewValue = null)
    {
        return (bool)$this->updateValue("AddAWStatsScript", $NewValue);
    }

    /**
     * Gets or sets the display limits by default system configuration setting.
     * Used by the Advanced search page to determine whether search limits should
     * be displayed or hidden.
     * @param bool $NewValue The new display limits by default system
     *       configuration setting.  (OPTIONAL)
     * @return bool The display limits by default system configuration setting.
     */
    public function displayLimitsByDefault(bool $NewValue = null)
    {
        return (bool)$this->updateValue("DisplayLimitsByDefault", $NewValue);
    }

    /**
     * Gets or sets the field display folder system configuration setting.
     * Used to get the sort order of metadata fields for display.
     * @param int $NewValue The new field display folder system configuration
     *       setting.  (OPTIONAL)
     * @return int The field display folder system configuration setting.
     */
    public function fieldDisplayFolder(int $NewValue = null)
    {
        return (int)$this->updateValue("FieldDisplayFolder", $NewValue);
    }

    /**
     * Gets or sets the field edit folder system configuration setting.
     * Used to get the sort order of metadata fields for editing.
     * @param int $NewValue The new field edit folder system configuration
     *       setting.  (OPTIONAL)
     * @return int The field edit folder system configuration setting.
     */
    public function fieldEditFolder(int $NewValue = null)
    {
        return (int)$this->updateValue("FieldEditFolder", $NewValue);
    }

    /**
     * Gets or sets the collapse metadata field groups system configuration
     * setting. Used by the MetadataFieldOrdering page to determine whether
     * metadata field groups should be collapsed.
     * @param bool $NewValue The new collapse metadata field groups system
     *       configuration setting.  (OPTIONAL)
     * @return bool The collapse metadata field groups system configuration setting.
     */
    public function collapseMetadataFieldGroups(bool $NewValue = null)
    {
        return (bool)$this->updateValue("CollapseMetadataFieldGroups", $NewValue);
    }

    /**
     * Gets or sets the require email with feedback system configuration setting.
     * @param bool $NewValue The new require email with feedback system
     *       configuration setting.  (OPTIONAL)
     * @return bool The require email with feedback system configuration setting.
     */
    public function requireEmailWithFeedback(bool $NewValue = null)
    {
        return (bool)$this->updateValue("RequireEmailWithFeedback", $NewValue);
    }

    /**
     * Gets or sets the search engine update priority system configuration
     *       setting.
     * @param int $NewValue The new search engine update priority system
     *       configuration setting.  (OPTIONAL)
     * @return int The search engine update priority system configuration setting.
     */
    public function searchEngineUpdatePriority(int $NewValue = null)
    {
        return (int)$this->updateValue("SearchEngineUpdatePriority", $NewValue);
    }

    /**
     * Gets or sets the recommender engine update priority system configuration
     *       setting.
     * @param int $NewValue The new recommender engine update priority system
     *       configuration setting.  (OPTIONAL)
     * @return int The recommender engine update priority system configuration
     *       setting.
     */
    public function recommenderEngineUpdatePriority(int $NewValue = null)
    {
        return (int)$this->updateValue("RecommenderEngineUpdatePriority", $NewValue);
    }

    /**
     * Gets or sets the has registered system configuration setting.
     * @return bool The has registered system configuration setting.
     */
    public function hasRegistered()
    {
        return strlen($this->updateValue("LastRegisteredOn", null)) ? true : false;
    }

    /**
     * Gets or sets the email line ending system configuration setting.
     * @param string $NewValue The new email line ending system configuration
     *       setting.  (OPTIONAL)
     * @return string The email line ending system configuration setting.
     */
    public function emailLineEnding(string $NewValue = null)
    {
        return $this->updateValue("EmailLineEnding", $NewValue);
    }

    /**
     * Gets or sets the default user privileges system configuration setting.
     * @param array $NewValue The new default user privileges system
     *       configuration setting.  (OPTIONAL)
     * @return array The default user privileges system configuration setting.
     */
    public function defaultUserPrivs(array $NewValue = null)
    {
        if ($NewValue !== null) {
            $NewValue = serialize($NewValue);
        }

        $Result = unserialize($this->updateValue("DefaultUserPrivs", $NewValue));

        if (!is_array($Result)) {
            $Result = $this->defaultUserPrivs([]);
        }

        return $Result;
    }

    /**
     * Gets or sets the root URL override system configuration setting.
     * @param string $NewValue The new root URL override system
     *       configuration setting.  (OPTIONAL)
     * @return string The root URL override system configuration setting.
     */
    public function rootUrlOverride(string $NewValue = null)
    {
        return $this->updateValue("RootUrlOverride", $NewValue);
    }

    /**
     * Gets or sets the prefer HTTP_HOST system configuration setting.
     * @param bool $NewValue The new prefer HTTP_HOST system
     *       configuration setting.  (OPTIONAL)
     * @return bool The prefer HTTP_HOST system configuration setting.
     */
    public function preferHttpHost(bool $NewValue = null)
    {
        return (bool)$this->updateValue("PreferHttpHost", $NewValue);
    }

    /**
     * Gets or changes a value in the database.
     * @param string $FieldName Name of database field.
     * @param mixed $NewValue New value to set.  (OPTIONAL)
     * @return mixed Requested value.
     */
    public function value($FieldName, $NewValue = null)
    {
        return $this->updateValue($FieldName, $NewValue);
    }

    /**
     * Gets or sets Show Group Names system configuration setting
     * @param bool $NewValue The new Show Group Names system configuration
     *       setting.  (OPTIONAL)
     * @return bool The Show Group Names system configuration setting.
     */
    public function showGroupNamesEnabled(bool $NewValue = null)
    {
        return (bool)$this->updateValue("ShowGroupNamesEnabled", $NewValue);
    }

    /**
     * Get/set the PasswordMinLength setting.
     * @param int $NewValue Updated value (OPTIONAL)
     * @return int current value.
     */
    public function passwordMinLength(int $NewValue = null)
    {
        return (int)$this->updateValue("PasswordMinLength", $NewValue);
    }

    /**
     * Get/set the PasswordUniqueChars setting.
     * @param int $NewValue Updated value (OPTIONAL)
     * @return int current value.
     */
    public function passwordUniqueChars(int $NewValue = null)
    {
        return (int)$this->updateValue("PasswordUniqueChars", $NewValue);
    }

    /**
     * Get/set the PasswordRequiresPunctuation setting.
     * @param bool $NewValue Updated value (OPTIONAL)
     * @return bool current value.
     */
    public function passwordRequiresPunctuation(bool $NewValue = null)
    {
        return (bool)$this->updateValue("PasswordRequiresPunctuation", $NewValue);
    }

    /**
     * Get/set the PasswordRequiresMixedCase setting.
     * @param bool $NewValue Updated value (OPTIONAL)
     * @return bool current value.
     */
    public function passwordRequiresMixedCase(bool $NewValue = null)
    {
        return (bool)$this->updateValue("PasswordRequiresMixedCase", $NewValue);
    }

    /**
     * Get/set the PasswordRequiresDigits setting.
     * @param bool $NewValue Updated value (OPTIONAL)
     * @return bool current value.
     */
    public function passwordRequiresDigits(bool $NewValue = null)
    {
        return (bool)$this->updateValue("PasswordRequiresDigits", $NewValue);
    }

    /**
     * Get/set the ID for the mailer template used for password change emails.
     * @param int $NewValue Updated value (OPTIONAL)
     * @return int|false current value, or false if null.
     */
    public function passwordChangeTemplateId(int $NewValue = null)
    {
        return $this->updateValue("PasswordChangeTemplateId", $NewValue);
    }

    /**
     * Get/set the ID for the mailer template used for account activation emails.
     * @param int $NewValue Updated value (OPTIONAL)
     * @return int|false current value, or false if null.
     */
    public function activateAccountTemplateId(int $NewValue = null)
    {
        return $this->updateValue("ActivateAccountTemplateId", $NewValue);
    }

    /**
     * Get/set the ID for the mailer template used for changing the account's associated email.
     * @param int $NewValue Updated value (OPTIONAL)
     * @return int|false current value, or false if null.
     */
    public function emailChangeTemplateId(int $NewValue = null)
    {
        return $this->updateValue("EmailChangeTemplateId", $NewValue);
    }

    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $DB;

    /**
     * Gets or changes a value in the database.
     * @param string $FieldName Name of database field.
     * @param mixed $NewValue The new value to set.  Use NULL to not set
     *       a new value (i.e. when just getting a value).
     * @param string $Table Name of database table.  (OPTIONAL)
     * @return mixed Requested value.
     */
    public function updateValue($FieldName, $NewValue, $Table = "SystemConfiguration")
    {
        return $this->DB->updateValue($FieldName, $NewValue);
    }
}
